﻿using System;
using System.Collections.Generic;
using System.Configuration;
using System.Device.Location;
using System.Linq;
using System.Net;
using System.Net.Http;
using System.Threading.Tasks;
using System.Web.Http;
using System.Web.Http.Description;
using System.Web.OData;
using System.Web.OData.Routing;
using BingMapsRESTToolkit;
using Microsoft.Web.Http;
using PpmsDataService.Models;
using PpmsDataService.ModelsEnumTypes;
using PpmsDataService.V1.Mappers;
using PpmsDataService.VA.PPMS.Context;
using VA.PPMS.Context;
using System.Net.Http.Headers;
using System.Runtime.Serialization.Json;
using System.Security.Cryptography.X509Certificates;
using System.Text;
using System.IO;

namespace PpmsDataService
{
    /// <summary>
    /// Provides unbound, utility functions.
    /// </summary>
    [ApiVersionNeutral]
    public class GlobalFunctionsController : ODataController
    {
        [HttpGet]
        [MapToApiVersion("1.0")]
        [ResponseType(typeof(Coordinates))]
        [ODataRoute("Geocode")]
        public async Task<HttpResponseMessage> Geocode([FromODataUri] string address)
        {
            using (var context = new PpmsContext(await PpmsContextHelper.GetProxy()))
            {
                string key = ConfigurationManager.AppSettings["BingMapsKey"];
                var request = new GeocodeRequest()
                {
                    Query = address,
                    IncludeIso2 = true,
                    IncludeNeighborhood = true,
                    MaxResults = 25,
                    BingMapsKey = key
                };

                //Process the request by using the ServiceManager.
                var response = await ServiceManager.GetResponseAsync(request);

                if (response != null &&
                    response.ResourceSets != null &&
                    response.ResourceSets.Length > 0 &&
                    response.ResourceSets[0].Resources != null &&
                    response.ResourceSets[0].Resources.Length > 0)
                {
                    var result = response.ResourceSets[0].Resources[0] as Location;
                    if (result != null)
                    {
                        double latitude = result.Point.Coordinates[0];
                        double longitude = result.Point.Coordinates[1];
                        var coordinate = new GeoCoordinate(latitude, longitude);


                        //Map the Lat, Long, and coordinate to Coordinates Class. 
                        var coordinates = new Coordinates();
                        coordinates.Latitude = latitude;
                        coordinates.Longitude = longitude;
                        coordinates.Coordinate = coordinate;

                        //Retrun the Coordinates
                        return Request.CreateResponse(coordinates);
                    }
                }

                var message = string.Format("Unable to Geocode the given address.");
                HttpError err = new HttpError(message);
                return Request.CreateErrorResponse(HttpStatusCode.NotFound, err);
            }
        }

        [HttpGet]
        [MapToApiVersion("1.0")]
        [ResponseType(typeof(AddressData))]
        [ODataRoute("ValidateAddress")]      
        public async Task<HttpResponseMessage> ValidateAddress([FromODataUri] string streetAddress, [FromODataUri] string city, [FromODataUri] string state, [FromODataUri] string zip)
        {
            ServicePointManager.ServerCertificateValidationCallback += (sender, certificate, chain, sslPolicyErrors) => true;
            ServicePointManager.SecurityProtocol = SecurityProtocolType.Tls12;
            try
            {
                //This Controller Action called the Address Validation API from Vets360
                using (var client = GetHttpClient())
                {
                    client.BaseAddress = new Uri("https://int.vet360.DNS   ");
                    //client.BaseAddress = new Uri("https://dev.vet360.DNS   ");

                    var payload = new RootObjectRequest
                    {
                        requestAddress = new Models.Address
                        {
                            addressLine1 = streetAddress,
                            city = city,
                            stateProvince = new StateProvince
                            {
                                name = state
                            },
                            zipCode5 = zip
                        }
                    };

                    var json = Serializee(payload);

                    var content = new StringContent(json, Encoding.UTF8, "application/json");
                    content.Headers.ContentType = new MediaTypeHeaderValue("application/json");

                    var response = client.PostAsync("/address-validation/address/v1/validate", content).GetAwaiter().GetResult();

                    if (response.IsSuccessStatusCode)
                    {
                        var result = response.Content.ReadAsStringAsync().GetAwaiter().GetResult();
                        if (string.IsNullOrEmpty(result))
                        {

                        }
                        else
                        {
                            var addressValidationResult = Deserialize<RootObjectResponse>(result);
                            var addressData = await AddressDataMap.MapAddressData(addressValidationResult);
                            return Request.CreateResponse(addressData);
                        }
                    }
                    var addressValidationError = string.Format("Address Validation Unsuccessful: " + response.StatusCode.ToString());
                    HttpError addErr = new HttpError(addressValidationError);
                    return Request.CreateErrorResponse(HttpStatusCode.NotFound, addErr);

                }
            }
            catch (WebException we)
            {
                var addressValidationError = we.ToString();              
                HttpError addErr = new HttpError(addressValidationError);
                return Request.CreateErrorResponse(HttpStatusCode.NotFound, addErr);
             
            }
            catch (HttpRequestException he)
            {
                var addressValidationError = he.ToString();
                HttpError addErr = new HttpError(addressValidationError);
                return Request.CreateErrorResponse(HttpStatusCode.NotFound, addErr);
            }
            catch (Exception e)
            {
                var addressValidationError = e.ToString();               
                HttpError addErr = new HttpError(addressValidationError);
                return Request.CreateErrorResponse(HttpStatusCode.NotFound, addErr);
            }
            

        }

        [HttpGet]
        [MapToApiVersion("1.0")]
        [ResponseType(typeof(ProviderLocatorResult))]
        [ODataRoute("ProviderLocator")]
        
        public async Task<HttpResponseMessage> ProviderLocator([FromODataUri] string address, [FromODataUri] int radius, [FromODataUriAttribute] string specialtycode, [FromODataUriAttribute] int network, [FromODataUriAttribute] int gender, [FromODataUriAttribute] int primarycare, [FromODataUriAttribute] int acceptingnewpatients )
        {
            using (var context = new PpmsContext(await PpmsContextHelper.GetProxy()))
            {
                string key = ConfigurationManager.AppSettings["BingMapsKey"];

                //Geocode the Starting Address first
                var request = new GeocodeRequest()
                {
                    Query = address,
                    IncludeIso2 = true,
                    IncludeNeighborhood = true,
                    MaxResults = 25,
                    BingMapsKey = key
                };

                //Process the request by using the ServiceManager.
                var response = await ServiceManager.GetResponseAsync(request);

                if (response != null &&
                    response.ResourceSets != null &&
                    response.ResourceSets.Length > 0 &&
                    response.ResourceSets[0].Resources != null &&
                    response.ResourceSets[0].Resources.Length > 0)
                {
                    var result = response.ResourceSets[0].Resources[0] as Location;
                    if (result != null)
                    {
                        double startLatitude = result.Point.Coordinates[0];
                        double startLongitude = result.Point.Coordinates[1];
                        var startingCoord = new Coordinate(startLatitude, startLongitude);
                        var startingWayPoint = new SimpleWaypoint(startingCoord);
                        var origin = new List<SimpleWaypoint> { startingWayPoint };
    
                        var provServices = from ps in context.ppms_providerserviceSet
                                           where ps.StateCode.Value ==  (int)ppms_providerserviceState.Active
                                           where ps.ppms_network != null
                                           where ps.ppms_ProviderId != null
                                           where ps.ppms_specialty != null
                                           where ps.ppms_CareSiteAddressLatitude.Contains(".")
                                           where ps.ppms_CareSiteAddressLongitude.Contains(".")
                                           select ps;

                        //Add Optional Filters
                        
                        //Specialty
                        if (specialtycode != "none")
                            provServices = provServices.Where(ps => ps.ppms_specialtycode.Equals(specialtycode));
                        //Network Switch
                        var providerNetwork = NetworkIds.GetNetwork(network);
                        if(providerNetwork.Number != (int)Network.AnyNetwork)
                            provServices = provServices.Where(ps => ps.ppms_network.Id == providerNetwork.Id);
                      
                        switch (gender)
                        {
                            case 0:
                                //Any gender
                                break;
                            case 1:
                                //Male
                                provServices = provServices.Where(ps => ps.ppms_providergender.Equals(ppms_Gender.Male));
                                break;
                            case 2:
                                //Female
                                provServices = provServices.Where(ps => ps.ppms_providergender.Equals(ppms_Gender.Female));
                                break;
                            default:
                                break;
                        }

                        switch (primarycare)
                        {
                            case 0:
                                //No Preference
                                break;
                            case 1:
                                provServices = provServices.Where(ps => ps.ppms_providerisprimarycare.Equals(true));
                                break;
                            case 2:
                                provServices = provServices.Where(ps => ps.ppms_providerisprimarycare.Equals(false));
                                break;
                            default:
                                break;
                        }

                        switch (acceptingnewpatients)
                        {
                            case 0:
                                //No Preference
                                break;
                            case 1:
                                provServices = provServices.Where(ps => ps.ppms_provideracceptingnewpatients.Equals(true) && ps.ppms_provideracceptingva.Equals(true));
                                break;
                            case 2:
                                provServices = provServices.Where(ps => ps.ppms_provideracceptingnewpatients.Equals(false) && ps.ppms_provideracceptingva.Equals(false));
                                break;
                            default:
                                break;
                        }

                        

                        var ppmsProviderServicesInRadius = provServices
                          .Select(psr => new ProviderInRadius
                          {
                              ProviderService = psr,
                              Distance = Haversine.GetDistance(startLatitude, startLongitude
                             , Convert.ToDouble(psr.ppms_CareSiteAddressLatitude), Convert.ToDouble(psr.ppms_CareSiteAddressLongitude))
                          }).AsEnumerable();

                        //var ppmsProviderServicesList = ppmsProviderServicesInRadius.OrderBy(psr => psr.Distance).Take(100).Select(x => x.ProviderService).ToList();
                        var ppmsProviderServicesList = ppmsProviderServicesInRadius.Where(psr => psr.Distance <= radius).OrderBy(psr => psr.Distance).Take(100).Select(x => x.ProviderService).ToList();
                        if (ppmsProviderServicesList.Count > 0)
                        {

                            //Split up into separate lists of max 625 for the Distance Matrix Request.
                            //If adding a Start time the max # in a request is 100
                            //var ppmsProviderServicesParsed = ppmsProviderServices.Select((x, i) => new { Index = i, Value = x }).GroupBy(x => x.Index / 100).Select(x => x.Select(v => v.Value).ToList()).ToList();

                            var provLocatorList = new List<ProviderLocatorResult>();
                           // foreach (var provServiceList in ppmsProviderServicesParsed)
                           // {
                                //Create the List of Destinations
                                var destinationsList = new List<SimpleWaypoint>();
                                foreach (var ppmsProviderService in ppmsProviderServicesList)
                                {
                                    double latitude = Convert.ToDouble(ppmsProviderService.ppms_CareSiteAddressLatitude);
                                    double longitude = Convert.ToDouble(ppmsProviderService.ppms_CareSiteAddressLongitude);
                                    var coord = new Coordinate(latitude, longitude);
                                    var wayPoint = new SimpleWaypoint(coord);
                                    destinationsList.Add(wayPoint);
                                }
                                //Define the Distance Matrix Request
                                var distanceMatrixReq = new DistanceMatrixRequest()
                                {
                                    Origins = origin,
                                    BingMapsKey = key,
                                    DistanceUnits = DistanceUnitType.Miles,
                                    TimeUnits = TimeUnitType.Minute,
                                    Destinations = destinationsList,
                                    TravelMode = TravelModeType.Driving,
                                    //StartTime = DateTime.Now
                                };

                                //Process the request by using the ServiceManager.
                                var distMatrixResponse = await ServiceManager.GetResponseAsync(distanceMatrixReq);

                                if (distMatrixResponse != null &&
                                    distMatrixResponse.ResourceSets != null &&
                                    distMatrixResponse.ResourceSets.Length > 0 &&
                                    distMatrixResponse.ResourceSets[0].Resources != null &&
                                    distMatrixResponse.ResourceSets[0].Resources.Length > 0)
                                {
                                    if (distMatrixResponse.ResourceSets[0].Resources[0] is DistanceMatrix distanceMatrix)
                                    {
                                        var travelTimesDistances = distanceMatrix.Results;

                                        for (var i = 0; i < travelTimesDistances.Length; i++)
                                        {
                                            //Only do mapping if the Provider Service/Travel Distance falls within radius
                                            if (travelTimesDistances[i].TravelDistance <= radius)
                                            {
                                                var provLocatorResult = new ProviderLocatorResult();
                                                provLocatorResult.Miles = travelTimesDistances[i].TravelDistance;
                                                provLocatorResult.Minutes = travelTimesDistances[i].TravelDuration;
                                                provLocatorResult.ProviderName = ppmsProviderServicesList[i].ppms_providername;
                                                provLocatorResult.ProviderSpecialty = ppmsProviderServicesList[i].ppms_specialtynametext;
                                                provLocatorResult.SpecialtyCode = ppmsProviderServicesList[i].ppms_specialtycode;
                                                provLocatorResult.ProviderIdentifier = ppmsProviderServicesList[i].ppms_provideridentifer;
                                                provLocatorResult.WorkHours = ppmsProviderServicesList[i].ppms_workhours;
                                                
                                                if (ppmsProviderServicesList[i].ppms_qualityrankingtotalscore != null)
                                                {
                                                    provLocatorResult.QualityRanking = (int)ppmsProviderServicesList[i].ppms_qualityrankingtotalscore;
                                                }
                                                provLocatorResult.Latitude = Convert.ToDouble(ppmsProviderServicesList[i].ppms_CareSiteAddressLatitude);
                                                provLocatorResult.Longitude = Convert.ToDouble(ppmsProviderServicesList[i].ppms_CareSiteAddressLongitude);
                                                if (ppmsProviderServicesList[i].ppms_caresite != null)
                                                    provLocatorResult.CareSite = ppmsProviderServicesList[i].ppms_caresite.Name;
                                                provLocatorResult.CareSiteAddress = ppmsProviderServicesList[i].ppms_caresiteaddress + ' ' + ppmsProviderServicesList[i].ppms_caresitecity + ','+' '+
                                                    ppmsProviderServicesList[i].ppms_caresitestateprovince + ' ' + ppmsProviderServicesList[i].ppms_caresitezipcode;
                                                //Gender
                                                if (ppmsProviderServicesList[i].ppms_providergender != null)
                                                    switch (ppmsProviderServicesList[i].ppms_providergender.Value)
                                                    {
                                                        case (int)ppms_Gender.Male:
                                                            provLocatorResult.ProviderGender = ProviderGender.Male;
                                                            break;
                                                        case (int)ppms_Gender.Female:
                                                            provLocatorResult.ProviderGender = ProviderGender.Female;
                                                            break;
                                                        case (int)ppms_Gender.NotSpecified:
                                                            provLocatorResult.ProviderGender = ProviderGender.NotSpecified;
                                                            break;
                                                        case (int)ppms_Gender.Other:
                                                            provLocatorResult.ProviderGender = ProviderGender.Other;
                                                            break;
                                                    }

                                                if (ppmsProviderServicesList[i].ppms_qualityrankingtotalscore != null)
                                                    provLocatorResult.QualityRanking = (int)ppmsProviderServicesList[i].ppms_qualityrankingtotalscore;
                                                if (ppmsProviderServicesList[i].ppms_network != null)
                                                {
                                                    provLocatorResult.ProviderNetwork = ppmsProviderServicesList[i].ppms_network.Name;
                                                    var provNetwork = NetworkIds.GetNetwork(ppmsProviderServicesList[i].ppms_network.Id);
                                                    provLocatorResult.NetworkId = provNetwork.Number; 
                                                }
                                                if (ppmsProviderServicesList[i].ppms_providerisprimarycare != null)
                                                    provLocatorResult.ProviderPrimaryCare = ppmsProviderServicesList[i].ppms_providerisprimarycare.Value;
                                                if (ppmsProviderServicesList[i].ppms_provideracceptingnewpatients != null)
                                                    provLocatorResult.ProviderAcceptingNewPatients = ppmsProviderServicesList[i].ppms_provideracceptingnewpatients.Value;
                                            provLocatorList.Add(provLocatorResult);
                                            }
                                        }
                                    }
                                }
                            // }

                            //Sort the Results on Distance ascending.
                            provLocatorList.Sort((x, y) => x.Miles.CompareTo(y.Miles));
                            //Return the Results
                            return Request.CreateResponse(provLocatorList);

                        }
                        var message = string.Format("No Providers found based on Address, Radius and Filter Criteria given");
                        HttpError err = new HttpError(message);
                        return Request.CreateErrorResponse(HttpStatusCode.NotFound, err);
                    }             
                }
                var geoCodeErrorMessage = string.Format("Unable to Geocode the given address");
                HttpError geoErr = new HttpError(geoCodeErrorMessage);
                return Request.CreateErrorResponse(HttpStatusCode.NotFound, geoErr);
            }
        }

        public static class Haversine
        {
            public static bool IsInRadius(double lat1, double lon1, double lat2, double lon2, int radius)
            {
                var R = 6372.8; // In kilometers
                var dLat = toRadians(lat2 - lat1);
                var dLon = toRadians(lon2 - lon1);
                lat1 = toRadians(lat1);
                lat2 = toRadians(lat2);

                var a = Math.Sin(dLat / 2) * Math.Sin(dLat / 2) + Math.Sin(dLon / 2) * Math.Sin(dLon / 2) * Math.Cos(lat1) * Math.Cos(lat2);
                var c = 2 * Math.Asin(Math.Sqrt(a));
                var distance = R * 2 * Math.Asin(Math.Sqrt(a));
                if (distance <= radius)
                {
                    return true;
                }
                return false;
            }

            public static double GetDistance(double lat1, double lon1, double lat2, double lon2)
            {
                var R = 6372.8; // In kilometers
                var dLat = toRadians(lat2 - lat1);
                var dLon = toRadians(lon2 - lon1);
                lat1 = toRadians(lat1);
                lat2 = toRadians(lat2);
                var a = Math.Sin(dLat / 2) * Math.Sin(dLat / 2) + Math.Sin(dLon / 2) * Math.Sin(dLon / 2) * Math.Cos(lat1) * Math.Cos(lat2);
                var c = 2 * Math.Asin(Math.Sqrt(a));
                var distance = R * 2 * Math.Asin(Math.Sqrt(a));             
                return distance;
            }

            public static ppms_providerservice ReturnInRadius(ppms_providerservice provService, double lat1, double lon1, int radius)
            {
                double lat2 = Convert.ToDouble(provService.ppms_CareSiteAddressLatitude);
                double lon2 = Convert.ToDouble(provService.ppms_CareSiteAddressLongitude);
                var R = 6372.8; // In kilometers
                var dLat = toRadians(lat2 - lat1);
                var dLon = toRadians(lon2 - lon1);
                lat1 = toRadians(lat1);
                lat2 = toRadians(lat2);

                var a = Math.Sin(dLat / 2) * Math.Sin(dLat / 2) + Math.Sin(dLon / 2) * Math.Sin(dLon / 2) * Math.Cos(lat1) * Math.Cos(lat2);
                var c = 2 * Math.Asin(Math.Sqrt(a));
                var distance = R * 2 * Math.Asin(Math.Sqrt(a));
                provService.ppms_distance = distance;
                if (distance <= radius)
                {
                    return provService;
                }
                return null;
            }

            public static double toRadians(double angle)
            {
                return Math.PI * angle / 180.0;
            }         
        }

        public class ProviderInRadius
        {
            public ppms_providerservice ProviderService { get; set; }
            public double Distance { get; set; }
        }


        public static string Serializee<T>(T data)
        {
            var ms = new MemoryStream();
            var ser = new DataContractJsonSerializer(typeof(T));
            ser.WriteObject(ms, data);
            var json = ms.ToArray();
            ms.Close();

            return Encoding.UTF8.GetString(json, 0, json.Length);
        }

        public static T Deserialize<T>(string json)
        {
            var ms = new MemoryStream(Encoding.UTF8.GetBytes(json));
            var ser = new DataContractJsonSerializer(typeof(T));
            var result = (T)ser.ReadObject(ms);
            ms.Close();

            return result;
        }

        private static HttpClient GetHttpClient()
        {
            var clientHandler = new WebRequestHandler();
            clientHandler.ClientCertificates.Add(GetCertKeyVault());
            //clientHandler.ClientCertificates.Add(GetLocalCert());
            return new HttpClient(clientHandler);
        }

        public static X509Certificate2 GetLocalCert()
        {
            var store = new X509Store(StoreName.My, StoreLocation.LocalMachine);
            store.Open(OpenFlags.OpenExistingOnly | OpenFlags.ReadOnly);
            string certificateSubjectName = "CN=dws.ppms.DNS   , OU=PPMS, O=VA, L=Washington, S=DC, C=US";
            var cert = store.Certificates.Find(X509FindType.FindBySubjectDistinguishedName, certificateSubjectName, true);
            if (cert.Count < 1)
            {
                throw new Exception(string.Format("Could not find a valid client certificate with subject {0}", certificateSubjectName));
            }
            return cert[0];
        }

        private static X509Certificate2 GetCertKeyVault()
        {
            const string appId = "REDACTED";
            const string secret = "REDACTED";
            const string tenantId = "REDACTED";
            //const string certUri = "https://DNS.vault.usgovcloudapi.net/certificates/np-dws-ppms-va-gov-sslcert/REDACTED";
            const string certUri = "https://DNS.vault.usgovcloudapi.net/certificates/np-dws-ppms-nprod/REDACTED";

            var token = GetToken(appId, secret, tenantId);
            var cert = GetCertificateFromKeyVault(token.access_token, certUri);
            var privateKey = GetPrivateKeyKeyVault(token.access_token, cert.sid);

            //return new X509Certificate2(privateKey, (string)null);
            return new X509Certificate2(privateKey, (string)null, X509KeyStorageFlags.MachineKeySet | X509KeyStorageFlags.PersistKeySet | X509KeyStorageFlags.Exportable);
        }


        private static TokenResponse GetToken(string clientId, string clientSecret, string tenantId)
        {
            using (var httpClient = new HttpClient())
            {
                var formContent = new FormUrlEncodedContent(new[]
                {
                    new KeyValuePair<string, string>("resource", "https://vault.usgovcloudapi.net"),
                    new KeyValuePair<string, string>("client_id", clientId),
                    new KeyValuePair<string, string>("client_secret", clientSecret),
                    new KeyValuePair<string, string>("grant_type", "client_credentials")
                });

                var response = httpClient.PostAsync("https://login.windows.net/" + tenantId + "/oauth2/token", formContent).GetAwaiter().GetResult();

                return Deserialize<TokenResponse>(response.Content.ReadAsStringAsync().Result);
            }
        }

        public static CertificateResponse GetCertificateFromKeyVault(string token, string certificateUrl)
        {
            using (var httpClient = new HttpClient())
            {
                var request = new HttpRequestMessage(HttpMethod.Get, new Uri(certificateUrl + "?api-version=2016-10-01"));
                request.Headers.Authorization = new AuthenticationHeaderValue("Bearer", token);

                var response = httpClient.SendAsync(request).GetAwaiter().GetResult();

                return Deserialize<CertificateResponse>(response.Content.ReadAsStringAsync().GetAwaiter().GetResult());
            }
        }

        public static byte[] GetPrivateKeyKeyVault(string token, string certificateUrl)
        {
            using (var httpClient = new HttpClient())
            {
                var request = new HttpRequestMessage(HttpMethod.Get, new Uri(certificateUrl + "?api-version=2016-10-01"));
                request.Headers.Authorization = new AuthenticationHeaderValue("Bearer", token);
                var response = httpClient.SendAsync(request).GetAwaiter().GetResult();
                var privateKey = Deserialize<PrivateKeyResponse>(response.Content.ReadAsStringAsync().GetAwaiter().GetResult());
                return Convert.FromBase64String(privateKey.value);
            }
        }


    }

    public class TokenResponse
    {
        public string token_type { get; set; }
        public string expires_in { get; set; }
        public string ext_expires_in { get; set; }
        public string expires_on { get; set; }
        public string not_before { get; set; }
        public string resource { get; set; }
        public string access_token { get; set; }
    }

    public class CertificateResponse
    {
        public string id { get; set; }
        public string kid { get; set; }
        public string sid { get; set; }
        public string x5t { get; set; }
        public string cer { get; set; }
    }

    public class PrivateKeyResponse
    {
        public string value { get; set; }
    }

}